/* Redraw frame buffer, give messages and handle input using
   stdout/stdin only */
#include "main.h"

/* Some global variables */
frame_type frame_buffer;
message_type message_buffer;

/* Text output size */
static byte x_min, y_min, x_max, y_max;

/* Helper IO functions */
#ifndef windos
static void set_io_mode (bool mode)
{
  static struct termios old, new;
  if (!mode)
    {
      tcsetattr (STDIN_FILENO, TCSANOW, &old);
      return;
    }
  tcgetattr (STDIN_FILENO, &old);
  new = old;
  new.c_lflag &= ~(ICANON);
  new.c_lflag &= ~(ECHO);
  tcsetattr (STDIN_FILENO, TCSANOW, &new);
}
#endif

/* Basic output function return length */
/* No really need for \0 terminated string */
static byte put_line_at (byte * str, byte x, byte y, byte attr, byte len)
{
  /* We use bytes for size and coordinates, so no lines, longer then 256 */
#ifdef windos
  byte conio_colors[16] =
    {
      BLACK,
      RED,
      GREEN,
      BROWN,
      BLUE,
      MAGENTA,
      CYAN,
      LIGHTGRAY,
      LIGHTRED,
      LIGHTGREEN,
      YELLOW,
      LIGHTBLUE,
      LIGHTMAGENTA,
      LIGHTCYAN,
      WHITE,
      WHITE
    };
#endif
  byte line [280];
  bool bright;
  u16 i, n = 0;
  byte back, color;
  color = attr % 16;
#ifndef windos
  /* Put control chars */
  back = attr / 16;
  bright = FALSE;
  if (color > 7)
    {
      color -= 7;
      bright = TRUE;
    }

  if (bright)
    sprintf (line, "\033[1m\033[%dm\033[%dm\033[%d;%dH",
      color + 30, back + 40, y, x);
  else
    sprintf (line, "\033[0m\033[%dm\033[%dm\033[%d;%dH",
      color + 30, back + 40, y, x);

  for (n = 0;line[n];n++);
#else
  textcolor (conio_colors[color]);
#endif

  for (i = 0; i < len; i++)
    {
      /* Scan for bad things */
      if (x + i > x_max) break;
      if (!str[i]) break;
      if (str[i] > 32)
	line [n+i] = str [i];
      else
	line [n+i] = 32;
    }
  line [n+i] = 0;
#ifndef windos
  write (1, line, n+i);
#endif

#ifdef windos
  gotoxy (x,y);
  cputs (line);
#endif
}

/* Draw a buffer, we assume, that it fits the screen. */
/* Use as long lines as possible */
void redraw (void)
{
  byte x,y,attr,len;
  byte * line;
  u16 todo = frame_buffer.need_update;
  u16 pos;

  pos = 0;
  while (todo)
    {
      /* find first char */

      while (!frame_buffer.changed[pos]) pos++;
      frame_buffer.changed[pos] = 0;

      /* start a line */
      line = frame_buffer.ch + pos;
      attr = frame_buffer.attr[pos];
      len = 0;
      y = pos / frame_buffer.x;
      x = pos % frame_buffer.x;
      /* scan a line */
      while (todo)
	{
	  frame_buffer.changed[pos] = 0;
	  pos ++;
	  len ++;
	  todo --;
	  /* Not changed */
	  if (!frame_buffer.changed[pos]) break;
	  /* Different color */
	  if (attr != frame_buffer.attr[pos]) break;
	  /* Next row */
	  if (!(pos % frame_buffer.x)) break;
	  if (pos >= frame_buffer.x * frame_buffer.y) break;
	}
      /* Put a line */
      put_line_at (line, x_min + x, y_min + y, attr, len);
      if (pos >= frame_buffer.x * frame_buffer.y) break;
    }
  frame_buffer.need_update = 0;
  return;
}

/* Draw a buffer in html format. */
/* Use as long lines as possible */
static void redraw_html (const char * name)
{
  const char * html_colors[16] =
    {
      "\"#000000\"",
      "\"#800000\"",
      "\"#008000\"",
      "\"#804000\"",
      "\"#000080\"",
      "\"#800080\"",
      "\"#008080\"",
      "\"#808080\"",
      "\"#FF0000\"",
      "\"#00FF00\"",
      "\"#FFFF00\"",
      "\"#0000FF\"",
      "\"#FF00FF\"",
      "\"#00FFFF\"",
      "\"#FFFFFF\"",
      "\"#FFFFFF\""
    };


  byte x,y,attr;
  bool first;
  FILE * f = fopen (name, "w+");

  fputs ("<HTML>\n<HEAD>\n<META>\n<TITLE>screenshot</TITLE>\n</HEAD>\n", f);
  fputs ("<BODY TEXT=\"#FFFFFF\" BGCOLOR=\"#000000\"><FONT COLOR=\"#FFFFFF\">", f);
  fputs ("<PRE><TT>", f);

  for (y = 0; y < frame_buffer.y; y++)
    {
      first = TRUE;
      for (x = 0; x < frame_buffer.x; x++)
	{
	  /* First char of the same color. */
	  if (first)
	    {
	      attr = frame_buffer.attr[x + y * frame_buffer.x] % 16;
	      fputs ("<FONT COLOR=", f);
	      fputs (html_colors [attr], f);
	      fputc ('>', f);
	      fputc (frame_buffer.ch[x + y * frame_buffer.x], f);
	      first = FALSE;
	      continue;
	    }
	  /* Same color. */
	  if (attr == frame_buffer.attr[x + y * frame_buffer.x] % 16)
	    fputc (frame_buffer.ch[x + y * frame_buffer.x], f);
	  /* New color. */
	  else
	    {
	      first = TRUE;
	      /* Hack, process again. */
	      x--;
	      fputs ("</FONT>", f);
	    }
	}
      /* Close line. */
      fputs ("</FONT>\n", f);
    }

  fputs ("</TT></PRE>\n</BODY>\n</HTML>\n", f);

  fclose (f);
  return;
}





void init_ui (void)
{
  int i, j;
#ifndef windos
  write (1, "\033[40m\033[?25l\033[2J", 15);
  set_io_mode (TRUE);
#endif
#ifdef windos
  clrscr ();
  _setcursortype (0);
#endif
  x_min = 1;
  y_min = 2;
  x_max = 80;
  y_max = 24;
  message_buffer.string = malloc (MSG_SIZE * 10);
  for (i = 0; i < 10; i++)
    {
      for (j = 0; j < MSG_SIZE - 1; j++)
	message_buffer.string [j + MSG_SIZE * i] = 0;
      message_buffer.attr [i] = 14;
//      message_buffer.string [MSG_SIZE * (i+1) - 20] = 0;
    }
  message_buffer.position = 0;
  /* Anything but top line is a buffer */
  /* Top line is used for input and messages */
  frame_buffer.x = 80;//66
  frame_buffer.y = 23;//22
  frame_buffer.need_update = 0;
  frame_buffer.ch = malloc (frame_buffer.x * frame_buffer.y);
  frame_buffer.attr = malloc (frame_buffer.x * frame_buffer.y);
  frame_buffer.changed = malloc (frame_buffer.x * frame_buffer.y);

  for (i = 0; i < 80*23; i++)
    {
      frame_buffer.ch[i] = 32;
      frame_buffer.attr[i] = 15;
      frame_buffer.changed[i] = 1;
      frame_buffer.need_update ++;
    }
  redraw ();

  return;
}

void close_ui (void)
{
  redraw_html ("screenshot.html");
#ifndef windos
  write (1, "\033[?25h\033[0m", 10);
  set_io_mode (FALSE);
#else
  _setcursortype (2);
#endif
  free (frame_buffer.ch);
  free (frame_buffer.attr);
  free (frame_buffer.changed);
  free (message_buffer.string);
}

static byte esc_sequence (byte ch)
{
  if (ch == 27)
    if (getchar () == 91)
      switch (getchar ())
	{
	  case 52: (void) getchar(); return '1';
	  case 66: return '2';
	  case 54: (void) getchar(); return '3';
	  case 68: return '4';
	  case 69: return '5';
	  case 67: return '6';
	  case 49: (void) getchar(); return '7';
	  case 65: return '8';
	  case 53: (void) getchar(); return '9';
	}
  return ch;
}

/* TODO: do key bindings, also handle UI specific commands (if any) */
byte command (void)
{
  byte ch = 255;
  while (ch == 255)
    {
      wait_msec (50);
#ifndef windos
      ch = getchar ();
#endif
#ifdef windos
      ch = getch ();
#endif
      /* TODO: Process something here (UI level commands) */
    }

  /* Movement sequences TODO: vi keys */
#ifdef windos
  return ch;
#endif
  return esc_sequence (ch);
}

void redraw_msg (void)
{
    byte line [MSG_SIZE + 1];
    byte * m = message_buffer.string;
    byte pos = message_buffer.position;
    byte space = MSG_SIZE - 1;
    byte msgs = 0, i, s, first, p;
    bool done;
    /* How much messages fit in line ? */
    i = pos;
    while (space && (msgs < 9))
      {
	/* Current message */
	if (i == 0) i = 9; /* Warp around */
	/* string size */
	for (s = 0; m[i * MSG_SIZE + s]; s++);
	s++;
	/* String fits */
	if (space > s)
	  {
	    space -= s;
	    first = i;
	    msgs ++;
	    i --;
	  }
	/* Dont fit */
	else space = 0;
      }

    if (!msgs) return; /* Should not happen */

    if (msgs > 1) {msgs --; first ++;}; /* Hack, TODO: fix */

    if (msgs == 9)  /* All fit */
      {
	str_copy (" ", line, MSG_SIZE, ' ');
	/* Clear line */
	put_line_at (line,  1, 1, 14, 80);
      }

    /* Now assemble line */
    p = 0;
    for (i = first, done = FALSE; ;)
      {
	/* string size */
	for (s = 0; m[i * MSG_SIZE + s]; s++);
	if (i == pos)
	  str_copy (m + i * MSG_SIZE, line, MSG_SIZE, ' ');
	else
	  str_copy (m + i * MSG_SIZE, line, MSG_SIZE, 0);
	/* Draw it */
	put_line_at (line,  1 + p, 1, message_buffer.attr [i], 80);
	if (done) break;
	p += s;
	i ++; /* next */
	i %= 10; /* Warp around */
	if (i == pos) done = TRUE; /* All done */
      }
}

/* We assume buffer is large enough */
/* TODO: add more things, at least we need menu selection, simply treating
   info as menu lines with first char as hotkey
   (or maybe capital letter or ~ before hotkey char etc.)*/
void ask_player (byte what, byte * info, byte * result)
{
  bool log = TRUE;
  byte line [MSG_SIZE + 1];

  if (what >= 100)
    {
      what -= 100;
      log = FALSE;
    }

  if (log && replay)
    {
      if (what == 0)
	result [0] = fgetc (log_file);
      if (what == 1)
	fgets (result, 80, log_file);
      return;
    }

  /* char */
  str_copy (info, line, 80, ' ');
  if (what == 0)
    {
      put_line_at (line, 1, 1, 14, 80);
      result [0] = command ();
      redraw_msg ();
    }
  /* string */
  if (what == 1)
    {
      put_line_at (line, 1, 1, 14, 80);
      fgets (result, 80, stdin);
      redraw_msg ();
    }

  if (log)
    {
      if (what == 0)
	fputc (result [0], log_file);
      if (what == 1)
	fputs (result, log_file);
    }

  return;
}

/* Improve later */
/* This function allways write/read at least one byte to replay file */
/* TODO: use info field */
bool set_target (target_type * t)
{
  char cmd;
  bool OK = FALSE;
  if (replay)
    {
      cmd = fgetc (log_file);
      if (cmd != 'T') return FALSE;
      t->x0 = fgetc (log_file) - 32;
      t->y0 = fgetc (log_file) - 32;
      return TRUE;
    }

  put_line_at ("*", x_min + t->x0, y_min + t->y0, 1, 1);
  /* Mark for redraw later */
  if (!frame_buffer.changed [t->x0 + t->y0 * frame_buffer.x])
    frame_buffer.need_update ++;
  frame_buffer.changed [t->x0 + t->y0 * frame_buffer.x] = 1;

  while (1)
    {
      cmd = command ();
      if ((cmd == '\n') || (cmd == 't') || (cmd == ' '))
	{
	  OK = TRUE;
	  break;
	}
      else if ((cmd >= '1') && (cmd <= '9'))
	{
	  /* Clear mark */
	  redraw ();
	  /* Move pointer */
	  t->x0 += dx [cmd - '0'];
	  t->y0 += dy [cmd - '0'];
	  /* Check bounds */
	  if (t->x0 < t->x1) t->x0 = t->x1;
	  if (t->x0 > t->x2) t->x0 = t->x2;
	  if (t->y0 < t->y1) t->y0 = t->y1;
	  if (t->y0 > t->y2) t->y0 = t->y2;
	  put_line_at ("*", x_min + t->x0, y_min + t->y0, 1, 1);
	  /* Mark for redraw later */
	  if (!frame_buffer.changed [t->x0 + t->y0 * frame_buffer.x])
	    frame_buffer.need_update ++;
	  frame_buffer.changed [t->x0 + t->y0 * frame_buffer.x] = 1;
	}
      else
	{
	  OK = FALSE;
	  break;
	}
    }
  /* Clear mark */
  redraw ();
  if (!OK)
    {
      fputc ('F', log_file);
      return FALSE;
    }
  else
    {
      fputc ('T', log_file);
      fputc (t->x0+32, log_file);
      fputc (t->y0+32, log_file);
      return TRUE;
    }
}

/* Just put at top */
void give_msg (byte typ, byte * str)
{
  message_buffer.position ++;
  message_buffer.position %= 10;
  str_copy (str, message_buffer.string + message_buffer.position * MSG_SIZE,
	    MSG_SIZE, 0);
  message_buffer.attr [message_buffer.position] = typ;
  redraw_msg ();
  return;
}

void str_copy (byte * from, byte * to, u16 n, byte fill)
{
  u16 i;
  bool c = TRUE;
  for (i = 0; i < n-1; i++)
    {
      if (c & !from[i]) c = FALSE;
      if (!c && !fill)
	{
	  to [i] = 0;
	  break;
	}
      if (c)
        to [i] = from [i];
      else to [i] = fill;
    }
  if (fill) to [n] = 0;
  return;
}

#ifndef windos
void wait_msec (u16 w)
{
  struct timespec t;
  t.tv_sec = 0;
  t.tv_nsec = 1000000l * w;
  nanosleep (&t, NULL);
  return;
}
#endif

#ifdef windos
void wait_msec (u16 w)
{
  Sleep (w);
  return;
}
#endif