/* Various game data and save/load functions */
#include "main.h"
#include "data.h"

/* current chunk location */
byte map_x, map_y, map_z, map_t, num_e, num_f;

struct map_cell * world_map;

u16 num_fires = 0;

byte * view; /* Separate from map, only current view
		0 -- not seen, 1 -- recently seen, 2 -- visible, 3 -- shadow */

bool replay;

struct mob_type * mobs; /* Player controls mob #1 */

struct melee_attack ma_info [11] =
{
  /* for @, and other mobs */
  {100, 48,  80, 100, 0, 0}, /*0 undead normal */
  {100, 48,  80, 100, 2, 1}, /*1 undead slay */
  {110, 54,  90, 100, 0, 1}, /*2 undead rage */
  {110, 54,  90, 100, 2, 2}, /*3 undead slay + rage */
  /* for mobs */
  {100, 56,  80,  60, 0, 0}, /*4 troll */
  {100, 64,  64,  80, 0, 0}, /*5 (were) wolf */
  {110, 64,  80, 120, 0, 0}, /*6 Sword */
  {100, 60,  72, 100, 0, 0}, /*7 Axe */
  {100, 72,  60, 100, 0, 0}, /*8 Spear */
  {100, 50,  72,  80, 0, 0}, /*9 club */
  {110, 64,  80,  70, 1, 0}  /*10 Holy Sword, paladin only, very dangerous */
};

struct mob_race mr_info [MR_MAX] =
{
  {'@', 2, 10, 10, 4, 1, 0, 20, 0, 0, 0, 100}, /* normal */
  {'P',14, 10,  8, 6, 2, 0, 40,10, 8, 6, 100}, /* paladin */
  {'W', 7,  8, 10, 6, 0, 1, 20, 5, 5, 5, 100}, /* werewolf */
  {'T', 3, 10, 10, 0, 0, 1, 50, 4, 7, 9,  50}, /* troll */
  {'p',14,  8,  8, 4, 2, 0, 30, 6, 7, 8,  50}, /* fighter */
  {'o', 3,  8,  8, 0, 2, 0, 30, 9, 9, 9,  25}, /* caveman */
  {'w', 7,  6,  8, 0, 0, 1, 20, 5, 5, 5,  10}  /* wolf */
};

byte mob_num = 0;

const char dx [10] = {0,-1, 0, 1,-1, 0, 1,-1, 0, 1};
const char dy [10] = {0, 1, 1, 1, 0, 0, 0,-1,-1,-1};

/*********************** SIMPLE RNG ****************************/

#define RNG_a 1103515245
#define RNG_c 12345
/* lower 31 bits */
#define RNG_m 0x7FFFFFFF

u32 seed;

u16 rand_int (u16 x)
{
  seed = (RNG_a * seed + RNG_c) & RNG_m;
  return (seed >> 16) % x;
}


/************************* MAP RNG *****************************/
/* X0, y0 -- offset xs, ys -- size, map_rand is assumed to point to a buffer
   of xs*ys*4 bytes size */
struct mrng
{
  u32 map_seed, meta_seed;
  u32 *map_rand;
  u16 xs,ys;
  u16 x0,y0;
  bool new_map;
} static mrng;

/* Special RNG for map, depends on location */
/* x,y -- counts from x0,y0 */
static u16 rand_int_map (u16 d, u16 x, u16 y)
{
  /* No bounds check, be carefull */
  u16 offs;
  offs = x + y * mrng.xs;
  if (mrng.new_map)
    {
      u32 seed;
      seed = mrng.y0 + y;
      seed <<= 16;
      seed += mrng.x0 + x;
      seed += mrng.map_seed;
      mrng.map_rand [offs] = seed;
      return 0;
    }
  else
    {
      mrng.map_rand [offs] = (RNG_a * mrng.map_rand [offs] + RNG_c) & RNG_m;
      return (mrng.map_rand [offs] >> 16) % d;
    }
}

/* Meta RNG, get some number, dependent on coordinates, allways the same */
static u16 rand_meta (u16 d, u16 x, u16 y, bool next)
{
  static u32 seed;
  if (!next)
    {
      seed = y;
      seed <<= 16;
      seed += x;
      seed += mrng.meta_seed;
      /* Spin4 times */
      seed = (RNG_a * seed + RNG_c) & RNG_m;
      seed = (RNG_a * seed + RNG_c) & RNG_m;
      seed = (RNG_a * seed + RNG_c) & RNG_m;
    }
  seed = (RNG_a * seed + RNG_c) & RNG_m;
  return (seed >> 16) % d;
}

/* General map_gen ideas */
/* We shall have consistent terrain generator, making empty
   map (terrain only) of any size from any offset, so it is consistent.
   When, in addition to mob_list, etc we shall have a list of
   distant things (quest and friendly monsters, artifacts, towns, etc.)
   When map is scrolled, new part is generated, lists
   are patched, and things are moved to distant list (if worthy) or
   forgotten, also distant things are applyed as a patch to the map */

/* Consistent terrain generator */
enum map_t_type
{
  MT_FLOOR,
  MT_ROCK,
  MT_TREE,
  MT_WATER,
  MT_MAX,
  /* Extra types */
  MT_RUBBLE,
  MT_BUSH,
  MT_SWAMP
};

#define MD_MAX 4

struct metadata
{
  byte prob;
  byte start;
  byte end;
} static metadata [MD_MAX][MT_MAX] =
{
  {{ 0, 0, 0}, {15, 0, 9}, { 1, 5, 9}, { 1, 0, 0}}, /* Mountians */
  {{ 0, 0, 0}, { 1, 0, 2}, {20, 2, 9}, { 1, 0, 0}}, /* Woods */
  {{ 0, 0, 0}, { 6, 0, 0}, { 0, 0, 0}, {255,7, 9}}, /* Sea with rocks */
  {{ 0, 0, 0}, { 1, 0, 2}, { 1, 3, 7}, { 1, 0, 0}}  /* Plains */
};

/* Hack functions. This really should rely on some pre-defined large scale
   map. */
#define META_SIZE 8

/* Improve later to handle any s maybe */
u16 get_meta (u16 x, u16 y, byte f)
{
  static u16 mx = 0, my = 0;
  static u16 xs[5][4], ys[5][4], rs[5][4], ts[5][4];
  u16 idx, i, j, fx, fy, fr, ft;
  /* generate 5x4 field */
  if (f)
    {
      /* Starting grid */
      mx = x / META_SIZE - 1;
      my = y / META_SIZE - 1;
      for (i = 0; i < 5; i++)
	for (j = 0; j < 4; j++)
	  {
	    xs[i][j] = rand_meta (META_SIZE, mx + i, my + j, FALSE);
	    ys[i][j] = rand_meta (META_SIZE, mx + i, my + j, TRUE);
	    rs[i][j] = rand_meta (META_SIZE-2, mx + i, my + j, TRUE);
	    ts[i][j] = rand_meta (MD_MAX-1,  mx + i, my + j, TRUE);
	  }
    }

  idx = 3;/* Default */
  for (i = 0; i < 5; i++)
    for (j = 0; j < 4; j++)
      {
	fx = (mx + i) * META_SIZE + xs[i][j];
	fy = (my + j) * META_SIZE + ys[i][j];
        fr = rs[i][j];
	ft = ts[i][j];
	/* TODO: get closest, etc */
	if (fr >= distance (x,y,fx,fy))
	  {
	    idx = ft;
	    break;
	  }
      }
  /* Get idx */
  return idx;
}

/* Adjust (must be at least 1) */
#define RNG_SPIN 6
#define RNG_PASS 10

static byte * map_gen_t (u16 x0, u16 y0, u16 x1, u16 y1, u32 seed, u32 mseed)
{
  u16 xs,ys,x,y,i,mx,my,mi,rp,tp,wp,x2,y2;
  byte * buff;
  byte * res;

  /* Prepare stuff */
  xs = x1-x0+1;
  ys = y1-y0+1;
  res = malloc (xs*ys);

  xs += 2*RNG_PASS;
  ys += 2*RNG_PASS;
  buff = malloc (xs * ys);
  mrng.map_rand = malloc (xs * ys * 4);

  mrng.map_seed = seed;
  mrng.meta_seed = mseed;
  mrng.new_map = TRUE;
  mrng.x0 = x0;
  mrng.y0 = y0;
  mrng.xs = xs;
  mrng.ys = ys;

  for (i = 0; i < RNG_SPIN; i++)
    {
      if (i) mrng.new_map = FALSE;
      for (x = 0; x < xs; x++)
	for (y = 0; y < ys; y++)
	  (void)rand_int_map (1,x,y);
    }

  /* Clear map */
  for (x = 0; x < xs; x++)
    for (y = 0; y < ys; y++)
      buff [x + y * xs] = 0;

  /* Generate map */
  mx = x0 >> 3;
  my = y0 >> 3;
  mi = get_meta (mx, my, 1);
  for (i = 0; i < RNG_PASS; i++){
    for (x = 0; x < xs; x++)
      for (y = 0; y < ys; y++)
	{
	  /* Get meta index */
	  if ((((x0+x) >> 3) != mx) || (((y0+y) >> 3) != my))
	    {
	      mx = (x0+x) >> 3;
	      my = (y0+y) >> 3;
	      mi = get_meta (mx, my, 0);
	    }

	  //mi = 0; /* Test */

	  /* Add features */
	  /* Get probability */
	  if ((i >= metadata[mi][MT_WATER].start) &&
	      (i <= metadata[mi][MT_WATER].end))
	    wp = metadata[mi][MT_WATER].prob;
	  else
	    wp = 0;

	  if ((i >= metadata[mi][MT_TREE].start) &&
	      (i <= metadata[mi][MT_TREE].end))
	    tp = metadata[mi][MT_TREE].prob;
	  else
	    tp = 0;

	  if ((i >= metadata[mi][MT_ROCK].start) &&
	      (i <= metadata[mi][MT_ROCK].end))
	    rp = metadata[mi][MT_ROCK].prob;
	  else
	    rp = 0;

	  /* Skip non-floor grids TODO: maybe not */
	  if (buff [x + y * xs] != MT_FLOOR)
	    continue;

	  /* Calculate chance to put something */
	  for (x2 = x - 1; x2 <= x + 1; x2++)
	    for (y2 = y - 1; y2 <= y + 1; y2++)
	      {
		/* Warp around exploit !!!! */
		if (x2 >= xs) continue;
		if (y2 >= ys) continue;
		if (buff [x2 + y2 * xs] == MT_ROCK)
		  {
		    rp += 50;
		    tp += 20;
		    wp = 0;
		  }
		if (buff [x2 + y2 * xs] == MT_TREE)
		  {
		    tp += 40;
		    if (rp > 50) rp -= 50;
		    else rp = 0;
		  }
		if (buff [x2 + y2 * xs] == MT_WATER)
		  {
		    tp += 40;
		    rp = 0;
		    wp += 100;
		  }
	      }

	  /* put features */
	  if ((rand_int_map (1000, x, y) < wp))
	    buff [x + y * xs] = MT_WATER + 128;

	  else if ((rand_int_map (1000, x, y) < tp))
	    buff [x + y * xs] = MT_TREE + 128;

	  else if ((rand_int_map (1000, x, y) < rp))
	    buff [x + y * xs] = MT_ROCK + 128;
	}

      /* Newly added features are converted to old ones. */
      for (x = 0; x < xs; x++)
        for (y = 0; y < ys; y++)
	  if (buff [x + y * xs] >= 128) buff [x + y * xs] -= 128;
    }

  /* Copy map (with some modification) */

  for (x = 0; x <= x1-x0; x++)
    for (y = 0; y <= y1-y0; y++)
      {
	byte b;
	byte w = 0,t = 0,r = 0; /* Counters */
	b = buff [x + RNG_PASS + xs * (y+RNG_PASS)];
	/* Modify stuff */
	for (x2 = 0; x2 <= 2; x2++)
	  for (y2 = 0; y2 <= 2; y2++)
	    {
	      u16 p = x+x2-1 + RNG_PASS + xs * (y+y2-1+RNG_PASS);
	      if ((x2 == 1) && (y2 == 1)) continue;
	      /* Count nearby stuff */
	      if (buff [p] == MT_WATER) w++;
	      if (buff [p] == MT_TREE) t++;
	      if (buff [p] == MT_ROCK) r++;
	    }
	/* Surroundings completely overhelm a given tile */
	if (w > 4) b = MT_WATER;
	else if (t > 4) b = MT_TREE;
	else if (r > 4) b = MT_ROCK;
	/* Do some changes */
	else switch (b)
	  {
	    case MT_FLOOR:
	      if (r > 2) b = MT_RUBBLE;
	      else if (t > 2) b = MT_BUSH;
	      else if (w > 2) b = MT_SWAMP;
	    break;
	    case MT_WATER:
	      if (w < 2) b = MT_SWAMP;
	    break;
	    case MT_TREE:
	      if (t < 2) b = MT_BUSH;
	    break;
	    case MT_ROCK:
	      if (r < 2) b = MT_RUBBLE;
	    break;
	  }
	/* Write stuff */
	res [x + y * (x1-x0+1)] = b;
      }

  /* Free memory */
  free (buff);
  free (mrng.map_rand);
  return res;
}

void do_map_test (void)
{
  u16 x = 30000, y = 30000, i, j, k = 1, l;
  char str[40], cmd;
  byte * map;
  while (1)
    {
      sprintf (str, "MAP:(%d, %d, r:%d)", x, y, k);
      give_msg (2, str);
      /* Genenrate and draw a map from offset x,y and size 80x22 */
      if (!k) k = 1;
      for (l = 0; l < k; l++)
	{
      map = map_gen_t (x, y, x + 79, y + 21, 0, 0);
      /* Draw */
      for (i = 0; i < 80; i++)
	for (j = 0; j < 22; j++)
	  {
	    u16 p = i + j * 80;
	    frame_buffer.need_update++;
	    frame_buffer.changed[p] = 1;
	    switch (map[p])
	      {
		case MT_FLOOR:
		  frame_buffer.attr[p] = 14;
		  frame_buffer.ch[p] = '.';
		break;
		case MT_ROCK:
		  frame_buffer.attr[p] = 14;
		  frame_buffer.ch[p] = '#';
		break;
		case MT_TREE:
		  frame_buffer.attr[p] = 2;
		  frame_buffer.ch[p] = '+';
		break;
		case MT_WATER:
		  frame_buffer.attr[p] = 4;
		  frame_buffer.ch[p] = '=';
		break;
		case MT_RUBBLE:
		  frame_buffer.attr[p] = 14;
		  frame_buffer.ch[p] = ';';
		break;
		case MT_BUSH:
		  frame_buffer.attr[p] = 9;
		  frame_buffer.ch[p] = ':';
		break;
		case MT_SWAMP:
		  frame_buffer.attr[p] = 11;
		  frame_buffer.ch[p] = '~';
		break;
	      }
	  }
      redraw ();
      /* Free */
      free (map);
	}
      /* Repeat factior (testing performance) */
      cmd = command ();
      if (cmd == '+') {k *= 2; continue;}
      if (cmd == '-') {k /= 2; continue;}
      /* Update x and y */
      if ((cmd < '1') || (cmd > '9')) break;
      x += dx [cmd - '0'] * 5;
      y += dy [cmd - '0'] * 5;
    }
  return;
}

void add_mob (byte x, byte y)
{
  byte race = 255, i = 0, k;
  if (world_map [MAP_IDX (x, y)].terr == T_WALL) return;
  mob_num ++;
  if (mob_num == 1) race = 0;
  else
    while (race == 255)
      {
	i = randint (MR_MAX - 1);
	/* Use power as rarity */
	if (randint (mr_info[i].pow) <= 10)
	  race = i;
      }
  /* fill the record */
  mobs [mob_num].ch = mr_info[i].ch;
  mobs [mob_num].at = mr_info[i].at;
  mobs [mob_num].x = x;
  mobs [mob_num].y = y;
  mobs [mob_num].tx = 255;
  mobs [mob_num].ty = 255;
  mobs [mob_num].en = randint(25);
  mobs [mob_num].mt = 25;
  mobs [mob_num].hp = 6 * mr_info[i].str;
  mobs [mob_num].will = mr_info[i].mwill / 2;
  mobs [mob_num].r = race;
  k = randint(3);
  mobs [mob_num].ma = mr_info[i].a1;
  if (k == 2) mobs [mob_num].ma = mr_info[i].a2;
  if (k == 3) mobs [mob_num].ma = mr_info[i].a3;
  mobs [mob_num].stun = 0;
  mobs [mob_num].dot = 0;
  mobs [mob_num].reg = 0;
  world_map [MAP_IDX (x, y)].mob = mob_num;
  /* TODO: charges */
  if (mob_num == 1)
    {
      mobs [mob_num].en = 250;
      mobs [mob_num].fire_charges = 5;
      mobs [mob_num].ice_charges = 5;
      mobs [mob_num].throw_charges = 15;
      mobs [mob_num].heal_charges = 5;
      mobs [mob_num].haste_charges = 5;
    }
  else
    switch (race)
      {
	case 1:
	  mobs [mob_num].fire_charges = 5;
	break;
	case 2:
	  mobs [mob_num].ice_charges = 5;
	break;
	case 3:
	  mobs [mob_num].throw_charges = 15;
	break;
      }
  return;
}

static void map_gen (void)
{
  byte x,y,x1,y1,wp,tp;
  u16 w = DUN_S / 5, t = DUN_S / 3;
  for (x = 0; x < DUN_X; x++)
    for (y = 0; y < DUN_Y; y++)
      {
	if ((x == 0) || (y == 0) || (x == DUN_X - 1) || (y == DUN_Y - 1))
	  world_map [MAP_IDX (x, y)].terr = T_WALL;
	else
	  world_map [MAP_IDX (x, y)].terr = T_FLOOR;

	world_map [MAP_IDX (x, y)].mob = 0;
	world_map [MAP_IDX (x, y)].obj = 0;
      }

  /* Attempt to place cliffs / trees (simple) */
  while (t || w)
    {
      for (x = 1; x < DUN_X - 1; x++)
        for (y = 1; y < DUN_Y - 1; y++)
          {
	    wp = 3;
	    tp = 0;
	    /* Skip non-floor grids */
	    if (!(world_map [MAP_IDX (x, y)].terr == T_FLOOR))
	      continue;
	    /* Calculate chance to put something */
	    for (x1 = x - 1; x1 < x + 1; x1++)
	      for (y1 = y - 1; y1 < y + 1; y1++)
		{
		  if (world_map [MAP_IDX (x1, y1)].terr == T_WALL)
		    {
		      wp += 5;
		      tp ++;
		    }
		  if (world_map [MAP_IDX (x1, y1)].terr == T_TREE)
		    {
		      tp += 5;
		      if (wp > 5) wp -= 5;
		      else wp = 0;
		    }
		}
	    if ((randint (100) <= wp) && w)
	      {
		world_map [MAP_IDX (x, y)].terr = T_WALL;
		w --;
	      }
	    if ((randint (100) <= tp) && t)
	      {
		world_map [MAP_IDX (x, y)].terr = T_TREE;
		t --;
	      }
	  }
    }

  return;
}

/* TODO: write save / load (readable text for now) */
void init_game (void)
{
  u16 i;
  byte * str = "Replay ?";
  byte res [1];
  byte ch;

  /* TODO: default values to variables that are not saved */
  world_map = malloc (sizeof (struct map_cell) * MAP_S);
  for (i = 0; i < MAP_S; i++)
    {
      world_map [i].looks_ch = ' ';
      world_map [i].looks_at = 7;
    }
  mobs = malloc (sizeof (struct mob_type) * 256);
  view = malloc (DUN_S);
  for (i = 0; i < DUN_S; i++) view [i] = 0;
  /* So we loaded correctly */;
  if (load_game (save_file))
    {
      /* Open log to append */
      log_file = fopen ("log", "a"); /* TODO: file names */
      return;
    }

  /* Failed load */
  /* Open or create log */
  log_file = fopen ("log", "r");
  if (log_file)
    {
      ask_player (100, str, res);
      if ((res[0] == 'y') || (res[0] == 'Y'))
        {
          replay = TRUE;
          /* read seed */
          if (log_file)
	    {
	      unsigned long s;
	      fscanf (log_file, "Random SEED: %lx\n", &s);
	      seed = s;
	    }
          else
	    {
	      replay = FALSE; /* failure */
	      fclose (log_file);
	    }
        }
    }
  else replay = FALSE;

  /* New log, write seed */
  if (!replay)
    {
      log_file = fopen ("log", "w");
      fprintf (log_file, "Random SEED: %lx\n", (unsigned long)seed);
    }

  map_x = map_y = map_z = map_t = 0;
  mob_num = 0;
  map_gen ();
  for (i = 0; i < 15; i++)
    add_mob (randint (60) + 2, randint (16) + 3);
  check_win ();
  update ();
  /* Mega hack, replay TODO: stop key maybe */
  if (replay)
    {
      while (!feof(log_file))
	{
	  do_command (fgetc(log_file));
	  wait_msec (200);
	}
      /* Now done */
      replay = FALSE;
      /* Reopen log for append */
      fclose (log_file);
      log_file = fopen ("log", "a");
      /* Proceed normally */
    }
  return;
}

void close_game (void)
{
  /* TODO: save and quit (well, just save) */
  if (do_save) save_game (save_file);
  free (world_map);
  free (mobs);
  free (view);
  return;
}

void write_cell (FILE * f, struct map_cell * c)
{
  fprintf (f, "%d %d %d %d %d \n",
    c->looks_ch, c->looks_at, c->terr, c-> mob, c->obj);
}

void write_mob (FILE * f, struct mob_type * m)
{
  fprintf (f,
    "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d \n",
    m->ch, m->at, m->y, m->x, m->en, m->mt, m->heal_charges, m->haste_charges,
    m->ty, m->tx, m->fire_charges, m->ice_charges, m->throw_charges,
    m->hp, m->will, m->r, m->ma, m->stun, m->dot, m->reg);
}

void read_cell (FILE * f, struct map_cell * cell)
{
  int a,b,c,d,e;
  fscanf (f, "%d %d %d %d %d \n", &a, &b, &c, &d, &e);
  cell->looks_ch = a;
  cell->looks_at = b;
  cell->terr = c;
  cell-> mob = d;
  cell->obj = e;
}

void read_mob (FILE * f, struct mob_type * m)
{
  int a,b,c,d,e,g,h,i,j,k,l,n,o,p,q,r,s,t,u,v,w,x;
  fscanf (f,
  "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d \n",
	      &a, &b, &c, &d, &e, &g, &h, &i, &j, &k, &l, &n, &o, &p,
	      &r, &t, &u, &v, &w, &x);
  m->ch = a;
  m->at = b;
  m->y = c;
  m->x = d;
  m->en = e;
  m->mt = g;
  m->heal_charges = h;
  m->haste_charges = i;
  m->ty = j;
  m->tx = k;
  m->fire_charges = l;
  m->ice_charges = n;
  m->throw_charges = o;
  m->hp = p;
  m->will = r;
  m->r = t;
  m->ma = u;
  m->stun = v;
  m->dot = w;
  m->reg = x;
}


void save_game (FILE * f)
{

  byte x,y,i;
  fputs ("/* Draugr save file */ \n", f);
  /* Seed */
  fprintf (f, "Random SEED: %lx\n", (unsigned long) seed);
  /* Fires */
  fprintf (f, "Number of fires: %d\n", num_fires);
  /* Map */
  fputs ("/* Map info !!!! */ \n", f);
  for (x = 0; x < DUN_X; x++)
    for (y = 0; y < DUN_Y; y++)
      {
	write_cell (f, &world_map[MAP_IDX (x,y)]);
      }
  /* Mobs */
  fputs ("/* Mob info !!!! */ \n", f);
  fprintf (f, "mob_num: %d\n", mob_num);
  for (i = 1; i <= mob_num; i++)
    {
      write_mob (f, mobs + i);
    }

  return;
}

byte load_game (FILE * f)
{
  byte x,y,i;
  byte buff [80];
  int v;
  unsigned long s;
  if (!f)
    return FALSE;
  if (!fgets (buff, 80, f))
    return FALSE;
  map_x = map_y = map_z = 0;
  /* Seed */
  fscanf (f, "Random SEED: %lx\n", &s);
  seed = s;
  /* Fires */
  fscanf (f, "Number of fires: %d\n", &v);
  num_fires = v;
  /* Map */
  fgets (buff, 80, f);
  for (x = 0; x < DUN_X; x++)
    for (y = 0; y < DUN_Y; y++)
      {
	read_cell (f, &world_map[MAP_IDX (x,y)]);
      }
  /* Mobs */
  fgets (buff, 80, f);
  fscanf (f, "mob_num: %d\n", &v);
  mob_num = v;
  for (i = 1; i <= mob_num; i++)
    {
      read_mob (f, mobs + i);
    }
  update ();
  give_msg (11, "Loaded. ");
  return TRUE;
}
