/* Game functions */
#include "main.h"
#include "data.h"

/* Slay index 1 -- undead 2 -- human */
/* All measured in % more or less */

static byte view_radius = 4; /* Same for all things */
static byte last_hp = 0;
static byte proj_time = 100;
static bool proj_los = FALSE;


/*********************** MAP FUNCTIONS *************************/

/* Approximate distance, Angband style */
u16 distance (u16 x, u16 y, u16 x1, u16 y1)
{
  int dx = x1 - x, dy = y1 - y;
  int ax, ay, di;
  ax = dx > 0 ? dx : -dx;
  ay = dy > 0 ? dy : -dy;
  di = ax > ay ? ax + ay / 2 : ay + ax / 2;
  return di;
}

static bool on_map (byte x, byte y)
{
  if (x >= DUN_X) return FALSE;
  if (y >= DUN_Y) return FALSE;
  return TRUE;
}

/* Inner area */
static bool in_map (byte x, byte y)
{
  if (x >= DUN_X-1) return FALSE;
  if (y >= DUN_Y-1) return FALSE;
  if (x < 1) return FALSE;
  if (y < 1) return FALSE;
  return TRUE;
}

static void show_char (byte x, byte y, byte ch, byte at)
{
  if (!on_map (x, y)) return;
  frame_buffer.ch[MAP_COL + x + frame_buffer.x * (y + MAP_ROW)] = ch;
  frame_buffer.attr[MAP_COL + x + frame_buffer.x * (y + MAP_ROW)] = at;
  if (!frame_buffer.changed[MAP_COL + x + frame_buffer.x * (y + MAP_ROW)])
    frame_buffer.need_update ++;
  frame_buffer.changed[MAP_COL + x + frame_buffer.x * (y + MAP_ROW)] = 1;
  return;
}

static void show_mob (byte mob)
{
  byte x,y;
  x = mobs[mob].ox;
  y = mobs[mob].oy;
  /* Clear old location */
  show_char (x, y, world_map [MAP_IDX (x, y)].looks_ch,
		   world_map [MAP_IDX (x, y)].looks_at);
  /* Show mob */
  show_char (mobs[mob].x, mobs[mob].y, mobs[mob].ch, mobs[mob].at);
  return;
}

/* TODO: maybe some more suff */
static bool enemy (byte m, byte m1)
{
  /* Simplest, most logical and obvious way to handle friend or foe */
  /* Same coloured mobs are friends, differently coloured are foes */

  /* 'Nothing' (trap, etc.) can target everything */
  if (!m & m1) return TRUE;
  /* No target */
  if (!m1) return FALSE;
  /* Friend or foe */
  return mobs[m].at != mobs[m1].at;
}

static void show_friends ()
{
  byte i;
  for (i = 2; i <= mob_num; i++)
    {
      if (!enemy (1, i) &&
	  ((mobs[i].x != mobs[i].ox) || (mobs[i].y != mobs[i].oy)))
	show_mob (i);
      mobs[i].ox = mobs[i].x;
      mobs[i].oy = mobs[i].y;
    }
}

static void map_update (byte x1, byte y1, byte x2, byte y2)
{
  byte x,y;
  /* Simply copy map to buffer */
  for (x = x1; x <= x2; x++)
    for (y = y1; y <= y2; y++)
      {
	show_char (x, y, world_map [MAP_IDX (x, y)].looks_ch,
			 world_map [MAP_IDX (x, y)].looks_at);
      }
  /* Mobs */
  show_friends ();
  return;
}

/* Show map */
static void map_update_view (byte x, byte y, byte d)
{
  byte mob,lit,x1,y1,x2,y2,r;

  /* bounds */
  if (x > d) x1 = x-d;
  else x1 = 0;
  if (y > d) y1 = y-d;
  else y1 = 0;
  x2 = x + d;
  y2 = y + d;
  if (x2 >= DUN_X) x2 = DUN_X - 1;
  if (y2 >= DUN_Y) y2 = DUN_Y - 1;

  /* update */
  for (x = x1; x <= x2; x++)
    for (y = y1; y <= y2; y++)
      {
	/* extract light */
	/* viewable, redraw fully, terrain, then mob */
	if (view [x + DUN_X * y] == 2) lit = 1;
	else lit = 0;
	/* recently seen, redraw terain only, dark attribute */
	if (view [x + DUN_X * y] == 1) r = 1;
	else r = 0;
	/* Terrain TODO: all */
	if (lit || r)
	  switch (world_map [MAP_IDX (x, y)].terr)
	    {
	      case T_FLOOR:
		world_map [MAP_IDX (x, y)].looks_ch = '.';
		world_map [MAP_IDX (x, y)].looks_at = lit * 7 + 7;
	      break;
	      case T_WALL:
		world_map [MAP_IDX (x, y)].looks_ch = '#';
		world_map [MAP_IDX (x, y)].looks_at = lit * 7 + 7;
	      break;
	      case T_DOOR:
		world_map [MAP_IDX (x, y)].looks_ch = '+';
		world_map [MAP_IDX (x, y)].looks_at = lit * 7 + 3;
	      break;
	      case T_TREE:
		world_map [MAP_IDX (x, y)].looks_ch = '+';
		world_map [MAP_IDX (x, y)].looks_at = lit * 7 + 2;
	      break;
	      case T_FIRE:
		world_map [MAP_IDX (x, y)].looks_ch = '+';
		world_map [MAP_IDX (x, y)].looks_at = lit * 7 + 1;
	      break;
	    }
	/* Mobs */
	mob = world_map [MAP_IDX (x, y)].mob;
	if (mob && lit)
	  {
	    world_map [MAP_IDX (x, y)].looks_ch = mobs[mob].ch;
	    world_map [MAP_IDX (x, y)].looks_at = mobs[mob].at;
	  }
	/* forgot light */
	/* recently seen */
	if (view [x + DUN_X * y] == 2) view [x + DUN_X * y] = 1;
	else view [x + DUN_X * y] = 0;
      }
  map_update (x1, y1, x2, y2);
  redraw ();
  return;
}

/* Terrain damage, forest fires, etc. */
static bool damage_mob (byte m, byte d, byte m2);
static void map_stuff (void)
{
  byte x,y,i,x1,y1;

  /* Spawn mob */
  if ((randint (50) == 1) && (mob_num < 250))
    add_mob (randint (60) + 2, randint (16) + 3);

  /* Nothing is burning */
  if (!num_fires) return;

  /* Scan all over the map excluding outer walls for fires */
  for (x = 1; x < DUN_X - 1; x++)
    for (y = 1; y < DUN_Y - 1; y++)
      if (world_map[MAP_IDX(x,y)].terr == T_FIRE)
	{
	  /* burn a mob */
	  if (world_map[MAP_IDX(x,y)].mob)
	    damage_mob (world_map[MAP_IDX(x,y)].mob, 100 + 1, 0);
	  /* burning tree falls down, probably spreading fire */
	  if (randint(15) == 1)
	    {
	      world_map[MAP_IDX(x,y)].terr = T_FLOOR;
	      num_fires --;
	      /* Four burning branches fly out */
	      for (i = 0; i < 4; i++)
		{
		  x1 = x + randint (3) - 2;
		  y1 = y + randint (3) - 2;
		  if (world_map[MAP_IDX(x1,y1)].terr == T_TREE)
		    {
		      world_map[MAP_IDX(x1,y1)].terr = T_FIRE;
		      num_fires ++;
		    }
		}
	    }
	}
  return;
}

static void map_desc (byte x, byte y, byte * what)
{
  struct map_cell * c = &world_map[MAP_IDX(x,y)];
  if (c->mob == 1)
    {
      str_copy (what, "you", 4, 0);
      return;
    }
  if (c->mob == 1)
    {
      str_copy (what, "mob", 4, 0);
      return;
    }
  if (c->terr == T_WALL)
    {
      str_copy (what, "wall", 5, 0);
      return;
    }

  str_copy (what, "floor", 6, 0);
  return;

}

/******************* PROJECTION FUNCTIONS **********************/

/* fires 'typ' colored '*' from x,y to tx,ty up to dist range
   or until it hits something. Return actual range and put
   actually hit target location to x,y.
   Waits for proj_time ms delay for animation and assumes that map location
   with looks_ch > 32 is currently visible */
static byte proj (byte dist, byte typ, byte * x, byte * y, byte tx, byte ty)
{
  int dx = tx - *x, dy = ty - *y, sx = *x, sy = *y, d;
  int cx, cy, di;
  byte ch, at;
  bool hit = FALSE;
  byte shadow = 100;
  di = distance (sx, sy, tx, ty);
  if (!di) return 0;
  /* TODO: some fun projection algorythm maybe (as in Angband with
     hockey-sticks and trick shots) */
  /* just line */
  for (d = 1; d <= dist+1; d++)
    {
      /* Get position */
      *x = cx = sx + (dx * d) / di;
      *y = cy = sy + (dy * d) / di;

      /* Test collision */
      if ((cx != sx) || (cy != sy))
	{
	  if (world_map [MAP_IDX (cx, cy)].terr >= T_WALL)
	    hit = TRUE;
	  if (!proj_los && world_map [MAP_IDX (cx, cy)].mob)
	    hit = TRUE;
	  if (distance (sx, sy, cx, cy) >= dist)
	    hit = TRUE;
	}

      /* check bounds */
      if (cx < 0) cx = 0;
      if (cy < 0) cy = 0;
      if (cx >= DUN_X) cx = DUN_X-1;
      if (cy >= DUN_Y) cy = DUN_Y-1;

      /* LOS stuff */
      if (proj_los)
	{
	  /* visible */
	  if (d < shadow)
	    view [cx + cy * DUN_X] = 2;
	  /* not visible */
	  else if (d > shadow)
	    view [cx + cy * DUN_X] = 3;
	  /* LOS blocked, so next tiles are in shadow */
	  if (hit) shadow = d+1;
	}

      /* Show animation only over visible space */
      if (((cx != sx) || (cy != sy)) && !proj_los && typ &&
	  view [cx + DUN_X * cy] &&
	  (world_map [MAP_IDX (cx, cy)].looks_ch > 32))
	{
	  cx += MAP_COL;
	  cy += MAP_ROW;
	  ch = frame_buffer.ch[cx + frame_buffer.x * cy];
	  at = frame_buffer.attr[cx + frame_buffer.x * cy];

	  /* Show projection */
	  frame_buffer.ch[cx + frame_buffer.x * cy] = '*';
	  frame_buffer.attr[cx + frame_buffer.x * cy] = typ;
	  frame_buffer.changed[cx + frame_buffer.x * cy] = 1;
	  frame_buffer.need_update ++;
	  redraw ();

	  /* wait */
	  wait_msec (proj_time);

	  /* clear */
	  frame_buffer.ch[cx + frame_buffer.x * cy] = ch;
	  frame_buffer.attr[cx + frame_buffer.x * cy] = at;
	  frame_buffer.changed[cx + frame_buffer.x * cy] = 1;
	  frame_buffer.need_update ++;
	  redraw ();
	}

      if (hit && !proj_los) break;
    }

  return d;
}

/*********************** LOS FUNCTIONS *************************/

static void update_view (byte r, byte x, byte y)
{
  byte x1,y1,d,i,sx,sy;
  proj_los = TRUE;
  /* Move outwards in square pattern */
  for (d = 1; d <= r+1; d++)
    for (i = 0, x1 = x-d, y1 = y-d; i < d*8; i++)
      {
	/* grid is on map and not yet seen */
	if (on_map (x1, y1) && view [x1 + DUN_X * y1] < 2)
	  {
	    sx = x;
	    sy = y;
	    proj (r, 0, &sx, &sy, x1, y1);
	  }
	if (i <= d * 2) x1 ++;
	else if (i <= d * 4) y1 ++;
	else if (i <= d * 6) x1 --;
	else y1 --;
      }
  proj_los = FALSE;
  view [x + DUN_X * y] = 2;
  map_update_view (x, y, r+1);
  return;
}

static void py_view (void)
{
  update_view (view_radius, mobs[1].x, mobs[1].y);
  return;
}

/*********************** MOB FUNCTIONS *************************/

dam_view (m, dam, kill)
{
  byte str[2], attr, ch, at, cx, cy;
  cx = mobs[m].x;
  cy = mobs[m].y;
  if ((cx == 255) || (cy == 255)) return;
  if (view [cx + DUN_X * cy] != 1) return;
  sprintf (str, "%1d", dam);
  attr = 10;
  if (kill)
    {
      str [0] = 'X';
      attr = 8;
    }
  cx += MAP_COL;
  cy += MAP_ROW;
  ch = frame_buffer.ch[cx + frame_buffer.x * cy];
  at = frame_buffer.attr[cx + frame_buffer.x * cy];

  /* Show projection */
  frame_buffer.ch[cx + frame_buffer.x * cy] = str [0];
  frame_buffer.attr[cx + frame_buffer.x * cy] = attr;
  frame_buffer.changed[cx + frame_buffer.x * cy] = 1;
  frame_buffer.need_update ++;
  redraw ();

  /* wait */
  wait_msec (2 * proj_time);

  /* clear */
  frame_buffer.ch[cx + frame_buffer.x * cy] = ch;
  frame_buffer.attr[cx + frame_buffer.x * cy] = at;
  frame_buffer.changed[cx + frame_buffer.x * cy] = 1;
  frame_buffer.need_update ++;
  redraw ();

}

/* Check for kill m2 attack m using d */
static bool damage_mob (byte m, byte d, byte m2)
{
  u16 dam, stun, slay = 0, addhp, att, def, chance;
  bool kill = FALSE;
  byte z = 5,x,y;
  struct melee_attack * ma;
  struct mob_race * ar, * tr;
  x = mobs[m].x;
  y = mobs[m].y;
  if (!d) return FALSE;
  if (!m) return FALSE;
  if (m2)
    {
      ma = &ma_info[mobs[m2].ma];
      ar = &mr_info[mobs[m2].r];
    }

  tr = &mr_info[mobs[m].r];

  if (d == 1)  /* melee */
    {
      /* turn */
      mobs[m2].en -= 2500 / (int)(ma->speed + ar->dex);
      /* TODO: ranged damage */
      /* Melee */
      att = 100 - ar->AC / 2;
      def = 100 + (tr->AC * ma->AC_mult) / 100;
      chance = (ma->chance * ar->dex) / tr->dex;
      /* miss (TODO: message maybe) */
      if (randint(100) > chance) return FALSE;
      /* armor block */
      if (randint(def) > att) return FALSE;
      /* damage */
      dam = (ma->damage * ar->str) / 100;
      dam = (dam * att) / def;
      /* life drain */
      dam += ma->drain_hp;
      mobs[m2].hp += ma->drain_hp; /* can go above max */
      if (mobs[m2].hp > 200) mobs[m2].hp = 200;
      /* crits */
      if (chance > 100) chance -= 100;
      else chance = 0;
      /* slay */
      if (ma->slay == tr->slay) slay = 6;
      slay += (slay + dam) * chance / 50;
      /* stun */
      stun = (120 - ma->damage) * dam / ma->damage;
      mobs[m].stun += stun;
    }

  /* Ranged attacks */
  switch (d)
    {
      case 101: /* fire */
	dam = 8;
	slay = 0;
	mobs[m].dot += 8;
      break;
      case 102: /* nether */
	{
	  u16 wa,wt;
	  wa = mobs[m2].will;
	  wt = mobs[m].will;
	  dam = 6;
	  if (wa > wt)
	    dam = dam * 2 * wa * wa / (wa * wa + wt * wt);
	  if (wt > wa)
	    dam = dam * (wa * wa + wt * wt) / (2 * wt * wt);
	  z = 25; /* Necromantic power */
	  mobs[m2].hp += dam / 2; /* can go above max */
	  if (mobs[m2].hp > 200) mobs[m2].hp = 200;
	  if (tr->slay == 2) slay = 6;
	}
      break;
      case 103: /* boulder */
	{
	  u16 d;
	  d = distance (mobs[m2].x, mobs[m2].y, mobs[m].x, mobs[m].y);
	  if (d < 3) d = 3;
	  chance = 300 / d;
	  chance = (chance * ar->dex) / tr->dex;
	  if (chance < randint (100))
	    dam = 0;
	  else
	    dam = (ar->str * 100) / (tr->AC + 100);
	}
      break;
      case 106: /* ice */
	dam = 8;
	slay = 0;
	mobs[m].stun += 8;
	mobs[m].mt += 200 / (mobs[m].hp + 20 * mobs[m].will);
	if (mobs[m].mt > 50) mobs[m].mt = 50;
      break;
    }

  /* check kill */
  if ((dam+slay > mobs[m].hp / 4) && (dam+slay > rand_int (mobs[m].hp)))
    kill = TRUE;
  else (mobs[m].hp -= dam);

  if (d == 2) kill = TRUE; /* just kill */

  if ((m2 == 1) && (!kill))
    last_hp = (2 * mobs[m].hp) / mr_info[mobs[m].r].str;

  dam_view (m, dam, kill);

  if (!kill) return TRUE;

  /* Dead */
  if ((m2 == 1) && (randint (100) < tr->pow))
    {
      mobs[m2].will++;
      if (mobs[m2].will > ar->mwill)
	mobs[m2].will = ar->mwill;
    }
  if (m == 1)
    py_view ();
  /* Remove from the map */
  if ((m == 1) || (randint (tr->pow) > z) || (m2 != 1) || (d == 2))
    {
      world_map[MAP_IDX(mobs[m].x,mobs[m].y)].mob = 0;
      /* Send mob to better world, far from undead plague */
      mobs[m].x = 255;
      mobs[m].y = 255;
    }
  else /* Braaains!!! */
    {
      mobs[m].hp = tr->str * 6; /* Heal fully */
      mobs[m].at = 2; /* Zombie now */
      give_msg (2, "Braaains !!! ");
    }

  return TRUE;
}

static bool mob_moves_into_mob (byte m1, byte m2)
{
  /* TODO */
  if (!enemy (m1, m2)) return FALSE;

  if (m1 == 1)
    {
      if (damage_mob (m2, 1, m1))
	give_msg (2, "You hit a mob ! ");
      else
	give_msg (2, "You missed a mob ! ");
    }

  else if (m2 == 1)
    {
      if (damage_mob (m2, 1, m1))
	give_msg (1, "Mob hits you ! ");
      else
	give_msg (1, "Mob missed you ! ");
    }

  else damage_mob (m2, 1, m1);

  return TRUE;
}

static bool move_mob_to (byte mob, byte x, byte y)
{
  /* Hack, we swap mobs actually */
  byte mob2 = world_map [MAP_IDX (x, y)].mob;
  byte x2 = mobs[mob].x;
  byte y2 = mobs[mob].y;

  if ((world_map [MAP_IDX (x, y)].terr >= T_WALL) &&
      (world_map [MAP_IDX (x, y)].terr < T_DOOR))
    return FALSE;

  /* Maybe do something else instead (attack, etc.) */
  if (mob2)
    if (mob_moves_into_mob (mob, mob2)) return FALSE;

  world_map [MAP_IDX (x, y)].mob = mob;
  mobs[mob].x = x;
  mobs[mob].y = y;

  world_map [MAP_IDX (x2, y2)].mob = mob2;
  if (mob2)
    {
      mobs[mob2].x = x2;
      mobs[mob2].y = y2;
    }

  return TRUE;
}

/********************* MORE MOB FUNCTIONS **********************/

/* More or less closest visible and hostile target */
static byte closest_mob (byte x, byte y)
{
  byte x1,y1,d,i,sx,sy,r,m1,m2;
  r = view_radius;
  m1 = world_map [MAP_IDX(x, y)].mob;

  /* Move outwards in square pattern */
  for (d = 1; d <= r+1; d++)
    for (i = 0, x1 = x-d, y1 = y-d; i < d*8; i++)
      {
	m2 = world_map [MAP_IDX(x1, y1)].mob;

	/* grid is on map and here is a mob (TODO: check hostile, etc.) */
	if (on_map (x1, y1) && enemy (m1, m2))
	  {
	    sx = x;
	    sy = y;

	    /* Test projection */
	    proj (r, 0, &sx, &sy, x1, y1);
	    /* Hit the mob */

	    if ((sx == x1) && (sy == y1))
	      return m2;
	  }

	/* Next grid */
	if (i < d * 2) x1 ++;
	else if (i < d * 4) y1 ++;
	else if (i < d * 6) x1 --;
	else y1 --;
      }

  return 0;
}

/* One mob decides where to move */
/* Simple AI function. If the mob see an enemy it shoots (if can) or
   moves. If the mob does not see an enemy, but have seen, it moves to last
   seen location. Else it moves randomly */
void fire_proj (byte who, byte what);
static void move_mob (byte m)
{
  byte d,x,y,i,tm,e;

  x = mobs[m].x;
  y = mobs[m].y;

  /* See an enemy ? */
  tm = closest_mob (x, y);

  if (tm)
    {
      /* Set target */
      mobs[m].tx = mobs[tm].x;
      mobs[m].ty = mobs[tm].y;
      d = distance (x, y, mobs[tm].x, mobs[tm].y);
      /* fire if we can and want */
      if ((mobs[m].fire_charges) && (d > 1))
	{
	  fire_proj (m, 8);
	  mobs[m].fire_charges --;
	  return;
	}
      if ((mobs[m].ice_charges) && (d > 1))
	{
	  fire_proj (m, 13);
	  mobs[m].ice_charges --;
	  return;
	}
      if ((mobs[m].throw_charges) && (d > 1))
	{
	  fire_proj (m, 10);
	  mobs[m].throw_charges --;
	  return;
	}
    }

  /* We are at target, clear it. */
  if ((mobs[m].tx == x) && (mobs[m].ty == y))
    {
      mobs[m].tx = 255;
      mobs[m].ty = 255;
    }

  if ((mobs[m].tx != 255) && (mobs[m].tx != 255))
    {
      /* move to target */
      proj (1, 0, &x, &y, mobs[m].tx, mobs[m].ty);
      if ((x != mobs[m].x) && (y != mobs[m].y))
	e = mobs[m].mt + (mobs[m].mt / 5) * 2;
      else
	e = mobs[m].mt;
      if (move_mob_to (m, x, y))
        {
	  if (mobs[m].en > e) mobs[m].en -= e;
	  else mobs[m].en = 0;

	  if (world_map [MAP_IDX (x, y)].terr >= T_DOOR)
	    mobs[m].en -= 50;
	}
    }

  else
  /* Try random move 5 times*/
    for (i = 0; i < 5; i++)
      {
	d = randint(9);
	x = mobs[m].x + dx[d];
	y = mobs[m].y + dy[d];
	/* If we can move, use energy. */
	if (move_mob_to (m, x, y))
	  {
	    mobs[m].en -= mobs[m].mt;
	    if (dx[d] && dy[d])
	      {
		e = (mobs[m].mt / 5) * 2;
		if (mobs[m].en > e) mobs[m].en -= e;
		else mobs[m].en = 0;
	      }
	    if (world_map [MAP_IDX (x, y)].terr >= T_DOOR)
	      mobs[m].en -= 50;
	  }
	/* If no more energy stop here */
	if (mobs[m].en < 250) break;
      }

  return;
}

/* Do timed stuff with a mob */
static void relax (byte m)
{
  struct mob_race * r = &mr_info[mobs[m].r];
  if (mobs[m].reg)
    {
      mobs[m].hp ++;
      if (mobs[m].hp > r->str * 6)
	mobs[m].hp = r->str * 6;
      mobs[m].reg --;
    }
  if (mobs[m].dot)
    {
      if (mobs[m].hp) mobs[m].hp --;
      if (mobs[m].hp == 0)
	damage_mob (m, 2, 0);
      mobs[m].dot --;
    }
  if (mobs[m].stun) mobs[m].stun --;
  if (mobs[m].mt > 25) mobs[m].mt --;
  if (mobs[m].mt < 25) mobs[m].mt ++;
}

/* We assume that mob #1 already moved, so we move others, until
   mob #1 is ready to move again */
static void move_mobs ()
{
  byte i;
  /* Look at all mobs many times ! */
  while (1)
    {
      /* Local map time. Warps around a lot. */
      map_t ++;
      if (!(map_t % 10))
	map_stuff ();

      for (i = 1; i <= mob_num; i++)
	{
	  /* Skip dead mobs */
	  if ((mobs[i].x == 255) && (mobs[i].y == 255))
	    {
	      if (i==1) return; /* Urghh ! */
	      continue;
	    }

	  /* give energy +5 TODO: use stun as failure chance */
	  mobs[i].en += 500 / (100 +
	    (mobs[i].stun * 100) / mobs[i].hp + 20 * mobs[i].will);

	  if (!(map_t % 10))
	    relax (i);

	  /* Our @ */
	  if (i == 1)
	    {
	      /* @ have a turn */
	      if (mobs[1].en >= 250)
		return;
	      continue;
	    }

	  /* Move mob */
	  if (mobs[i].en >= 250)
	    move_mob (i);
	}
    }
}

bool check_win ()
{
  byte i;
  num_e = num_f = 0;
  for (i = 2; i <= mob_num; i++)
    {
      /* Skip dead. */
      if ((mobs[i].x == 255) || (mobs[i].y == 255)) continue;

      if (mobs[i].at == mobs[1].at) num_f++;
      else num_e++;
    }
  if (num_f >= num_e * 2) return TRUE;
  if (num_f >= 10) return TRUE;
  if (num_e) return FALSE;
  return TRUE;
}

/*********************** COMMANDS ******************************/
void fire_proj (byte who, byte what)
{
  byte x,y,tx,ty,m;
  byte *str = "Direction ? '5' to target nearest enemy. 't' target. ", res[1];

  x = mobs[who].x;
  y = mobs[who].y;
  tx = ty = 255;
  /* Ask for target */
  if (who == 1)
    {
      ask_player (0, str, res);
      if ((res[0] <= '9') && (res [0] >= '1'))
	{
	  if (res[0] != '5')
	    {
	      tx = x + dx [res[0] - '1' + 1];
	      ty = y + dy [res[0] - '1' + 1];
	    }
	}
      else if (res[0] == 't')
	{
	  target_type t;
	  t.x1 = MAP_COL + ((x > 8) ? x-8 : 1);
	  t.y1 = MAP_ROW + ((y > 8) ? y-8 : 1);
	  t.x2 = MAP_COL + ((x < DUN_X - 8) ? x + 8 : DUN_X-1);
	  t.y2 = MAP_ROW + ((y < DUN_Y - 8) ? y + 8 : DUN_Y-1);
	  m = closest_mob (x, y);
	  if (!m) m = 1;
	  t.x0 = MAP_COL+mobs[m].x;
	  t.y0 = MAP_ROW+mobs[m].y;
	  if (!set_target (&t)) return;
	  tx = t.x0-MAP_COL;
	  ty = t.y0-MAP_ROW;
	  /* Cannot shoot self */
	  if ((tx == x) && (ty == y)) return;
	}
      else return;
    }

  /* No target yet */
  if (tx == 255)
    {
      m = closest_mob (x, y);
      if (m)
	{
	  tx = mobs[m].x;
	  ty = mobs[m].y;
	}
      else return;
    }

  /* TODO: Trageting if who == 1 */
  /* TODO: maybe chance to miss or dodge */
  /* TODO: effects on hit */
  proj (10, what, &x, &y, tx, ty);

  mobs[who].en -= 45; /* fixed */

  damage_mob (world_map[MAP_IDX(x,y)].mob, 100 + what % 7, who);

  /* red projectile supposed to be fire */
  if ((what % 7 == 1) && (world_map[MAP_IDX(x,y)].terr == T_TREE))
    {
      world_map[MAP_IDX(x,y)].terr = T_FIRE;
      num_fires ++;
    }

  py_view ();

  return;
}

/*********************** STATUS ********************************/

static void w_str (byte attr, byte * str, byte x, byte y)
{
  byte i;
  for (i = 0; str [i]; i++)
    {
      frame_buffer.ch [x + i + y * frame_buffer.x] = str [i];
      frame_buffer.attr [x + i + y * frame_buffer.x] = attr;
      if (!frame_buffer.changed [x + i + y * frame_buffer.x])
	{
	  frame_buffer.changed [x + i + y * frame_buffer.x] = 1;
	  frame_buffer.need_update ++;
	}
    }
  return;
}

void update_status ()
{
  byte y = 0, i;
  byte line [15];
  byte * b = "             ";
  (void) check_win ();

  w_str (14, b, 0, y); y++;

  if (num_f)
    {
      sprintf (line, "Zombies:  %3d", num_f);
      w_str (2, line, 0, y);
    }
  else
    w_str (14, b, 0, y);
  y++;

  if (num_e)
    {
      sprintf (line, "Monsters: %3d", num_e);
      w_str (1, line, 0, y);
    }
  else
    w_str (14, b, 0, y);
  y++;


  w_str (14, b, 0, y); y++;

  sprintf (line, "Hp:       %3d", mobs[1].hp);
  if (mobs[1].hp >= mr_info[mobs[1].r].str * 3)
    w_str (2, line, 0, y);
  else
    w_str (1, line, 0, y);
  y++;

  sprintf (line, "Will:     %3d", mobs[1].will);
  w_str (2, line, 0, y); y++;

  w_str (14, b, 0, y); y++;

  if (mobs[1].fire_charges)
    {
      sprintf (line, "Fire bolt: %2d", mobs[1].fire_charges);
      w_str (8, line, 0, y);
    }
  else
    w_str (14, b, 0, y);
  y++;

  if (mobs[1].ice_charges)
    {
      sprintf (line, "Ice bolt:  %2d", mobs[1].ice_charges);
      w_str (13, line, 0, y);
    }
  else
    w_str (14, b, 0, y);
  y++;

  if (mobs[1].throw_charges)
    {
      sprintf (line, "Throw rock:%2d", mobs[1].throw_charges);
      w_str (10, line, 0, y);
    }
  else
    w_str (14, b, 0, y);
  y++;

  if (mobs[1].heal_charges)
    {
      sprintf (line, "Heal self: %2d", mobs[1].heal_charges);
      w_str (2, line, 0, y);
    }
  else
    w_str (14, b, 0, y);
  y++;

  if (mobs[1].haste_charges)
    {
      sprintf (line, "Speed up:  %2d", mobs[1].haste_charges);
      w_str (2, line, 0, y);
    }
  else
    w_str (14, b, 0, y);
  y++;

  if (mobs[1].will)
    {
      sprintf (line, "Nether bolt:%1d", mobs[1].will);
      w_str (9, line, 0, y);
    }
  else
    w_str (14, b, 0, y);
  y++;

  w_str (14, b, 0, y); y++;

  if (mobs[1].stun)
    {
      sprintf (line, "Stunned  :%3d", mobs[1].stun);
      w_str (10, line, 0, y); y++;
    }

  if (mobs[1].dot)
    {
      sprintf (line, "Burning  :%3d", mobs[1].dot);
      w_str (8, line, 0, y); y++;
    }

  if (mobs[1].reg)
    {
      sprintf (line, "Regenerate:%2d", mobs[1].reg);
      w_str (2, line, 0, y); y++;
    }

  sprintf (line, "Speed    :%3d", 2500 / mobs[1].mt);
  if (mobs[1].mt < 25)
    {
      w_str (13, line, 0, y);
      y++;
    }

  if (mobs[1].mt > 25)
    {
      w_str (10, line, 0, y);
      y++;
    }

  w_str (14, b, 0, y); y++;

  for (i = 0; i < 12; i++)
    if (i < last_hp)
      w_str (10, "*", i, y);
    else
      w_str (10, "-", i, y);

  y++;

  w_str (14, b, 0, y); y++;
  w_str (14, b, 0, y); y++;

  redraw ();

  redraw_msg ();

  return;
}

/********************* EXTERNAL FUNCTIONS **********************/

void update (void)
{
  /* TODO: status line, etc. */
  frame_buffer.need_update = 0;
  map_update (0, 0, DUN_X - 1, DUN_Y - 1);
  redraw ();
  py_view ();
  update_status ();
}

/* TODO: write some fun game here */
bool do_command (byte cmd)
{
  byte * str;
  byte res[1];
  byte x = mobs[1].x;
  byte y = mobs[1].y;
  byte en = mobs[1].en;
  byte e;
  last_hp = 0;
  switch (cmd)
    {
      case 32:
      case '5':
	mobs[1].en -= mobs[1].mt;
      break;
      case '1':
      case '2':
      case '3':
      case '4':
      case '6':
      case '7':
      case '8':
      case '9':
	/* move @ and use energy */
	x += dx[cmd - '1' + 1];
	y += dy[cmd - '1' + 1];
	if (dx[cmd - '1' + 1] && dy[cmd - '1' + 1]) e = mobs[1].mt +
	    (mobs[1].mt / 5) * 2;
	else e = mobs[1].mt;
	if (move_mob_to (1, x, y))
	  {
	    if (mobs[1].en > e) mobs[1].en -= e;
	    else mobs[1].en = 0;
	    if (world_map [MAP_IDX (x, y)].terr >= T_DOOR)
	      mobs[1].en -= 50;
	  }
      break;
      case 'f':
      case 'F':
	if (!mobs[1].fire_charges) break;
	mobs[1].fire_charges --;
	give_msg (8, "Fire bolt! ");
	fire_proj (1, 8);
      break;
      case 'i':
      case 'I':
	if (!mobs[1].ice_charges) break;
	mobs[1].ice_charges --;
	give_msg (13, "Ice bolt! ");
	fire_proj (1, 13);
      break;
      case 't':
      case 'T':
	if (!mobs[1].throw_charges) break;
	mobs[1].throw_charges --;
	give_msg (10, "You have thrown a boulder. ");
	fire_proj (1, 10);
      break;
      case 'n':
      case 'N':
	if (!mobs[1].will) break;
	mobs[1].will --;
	give_msg (9, "You invoke powers of death. ");
	fire_proj (1, 9);
      break;
      case 'h':
      case 'H':
	if (!mobs[1].heal_charges) break;
	mobs[1].heal_charges --;
	mobs[1].reg += 20;
	give_msg (2, "Regeneration. ");
	update_status ();
      break;
      case 's':
      case 'S':
	if (!mobs[1].haste_charges) break;
	mobs[1].haste_charges --;
	mobs[1].mt = 15;
	give_msg (2, "You speed up. ");
	update_status ();
      break;
    }
  if (mobs[1].en < en)
    {
      py_view ();

      wait_msec (100);

      /* Move others */
      move_mobs ();

      /* Dead */
      if ((mobs[1].x == 255) && (mobs[1].x == 255))
	{
	  str =  "You are destroyed !";
	  ask_player (100, str, res);
	  do_save = FALSE;
	  return FALSE;
	}

      py_view ();
      update_status ();

      if (check_win ())
	{
	  do_save = FALSE;
	  str = "/***** TOTAL WINNER ! *****/";
	  ask_player (0, str, res);
	  return FALSE;
	}
    }

  return TRUE;
}


