/*
 * File: main-wcn.c
 * Purpose: Provide native (i.e. non-curses) Win32 Console support
 *
 * This file written by Dean Anderson, based on the template by Ben Harrison
 *
 * Copyright (C) 2010 Dean Anderson
 * Parts of this code Copyright (c) 1997 Ben Harrison
 *
 * This work is free software; you can redistribute it and/or modify it
 * under the terms of either:
 *
 * a) the GNU General Public License as published by the Free Software
 *    Foundation, version 2, or
 *
 * b) the "Angband licence":
 *    This software may be copied and distributed for educational, research,
 *    and not for profit purposes provided that this copyright and statement
 *    are included in all such copies.  Other copyrights may also apply.
 */

#include "angband.h"

#ifdef USE_WCN


/* 
 * We need <windows.h> for the 'Sleep' function
 * and the Win32 Console functions
 */
#include <windows.h>

/*
 * This interface does not support sound, graphics or custom colours.
 *
 * Therefore the following compiler constants should NOT be
 * defined in 'config.h':
 *
 * USE_SOUND
 * USE_GRAPHICS
 * ALLOW_COLORS
 *
 * Leaving them defined will not stop this interface from 
 * compiling or working, but removing them will result in a
 * smaller and more efficient program, and prevent the
 * user being confused by having configuration options that
 * do nothing when tweaked.
 */

#include "main.h"


/*
 * Extra data to associate with each "window"
 *
 * Each "window" is represented by a "term_data" structure, which
 * contains a "term" structure, which contains a pointer (t->data)
 * back to the term_data structure.
 */
typedef struct term_data term_data;

struct term_data
{
	term t;
	HANDLE hIn;
	HANDLE hOut;
	HANDLE hError;
};

/*
 * Number of "term_data" structures to support XXX XXX XXX
 */
#define MAX_WCN_TERM 1


/*
 * An array of "term_data" structures, one for each "sub-window"
 */
static term_data data[MAX_WCN_TERM];




/*
 * An array of "color data" containing
 * a representation of the "angband_color_table" array
 * in the form used by Win32 console apps.
 */
static WORD color_data[MAX_COLORS];


/*
 * We need to trap the window being closed or CTRL_C being pressed.
 */
BOOL WINAPI TrapCtrlHandler( DWORD dwCtrlType )
{
	switch( dwCtrlType )
	{
	case CTRL_C_EVENT:
	case CTRL_CLOSE_EVENT:
		/* Save the game if valid */
		if(character_generated)
		{
			save_game();
		}
		quit(NULL);
	default:
		return FALSE; /* uhandled */
	}
}




/*** Function hooks needed by "Term" ***/


/*
 * Init a new "term"
 */
static void Term_init_wcn(term *t)
{
	COORD size;
	term_data *td = (term_data*)(t->data);

	/* XXX XXX XXX */

    td->hIn = GetStdHandle(STD_INPUT_HANDLE);
    td->hOut = GetStdHandle(STD_OUTPUT_HANDLE);
    td->hError = GetStdHandle(STD_ERROR_HANDLE);

	size.X=80;
	size.Y=25;
	SetConsoleScreenBufferSize(td->hOut,size);
}



/*
 * Nuke an old "term"
 */
static void Term_nuke_wcn(term *t)
{
	term_data *td = (term_data*)(t->data);
}



/*
 * Do a "user action" on the current "term"
 */
static errr Term_user_wcn(int n)
{
	term_data *td = (term_data*)(Term->data);

	/* We have no user actions */
	return (1);
}


/*
 * Do a "special thing" to the current "term"
 *
 * This function reacts to a large number of possible arguments, each
 * corresponding to a different "action request" by the "z-term.c" package,
 * or by the application itself.
 *
 * The "action type" is specified by the first argument, which must be a
 * constant of the form "TERM_XTRA_*" as given in "z-term.h", and the second
 * argument specifies the "information" for that argument, if any, and will
 * vary according to the first argument.
 *
 * In general, this function should return zero if the action is successfully
 * handled, and non-zero if the action is unknown or incorrectly handled.
 */
static errr Term_xtra_wcn(int n, int v)
{
	CONSOLE_CURSOR_INFO CursorInfo;
	COORD TopLeft = {0,0};
	DWORD EventCount;
	DWORD Written;
	INPUT_RECORD InRec;

	term_data *td = (term_data*)(Term->data);

	/* Analyze */
	switch (n)
	{
		case TERM_XTRA_EVENT:
		{
			/*
			 * Process some pending events XXX XXX XXX
			 *
			 * Wait for at least one event if "v" is non-zero
			 * otherwise, if no events are ready, return at once.
			 * When "keypress" events are encountered, the "ascii"
			 * value corresponding to the key is sent to the
			 * "Term_keypress()" function.  Certain "bizarre" keys,
			 * such as function keys or arrow keys, may send special
			 * sequences of characters, such as control-underscore,
			 * plus letters corresponding to modifier keys, plus an
			 * underscore, plus carriage return, which can be used by
			 * the main program for "macro" triggers.  This action
			 * should handle as many events as is efficiently possible
			 * but is only required to handle a single event, and then
			 * only if one is ready or "v" is true.
			 */

			/* If v == 0, we don't want to wait if we have no events */
			if(!v)
			{
				/* Find out how many events are pending */
				GetNumberOfConsoleInputEvents(td->hIn,&EventCount);
				/* If none, then quit, otherwise drop int event handling code */
				if(!EventCount)
				{
					return (0);
				}
			}

			ReadConsoleInput(td->hIn, &InRec, 1, &EventCount);
			switch(InRec.EventType)
			{
			case KEY_EVENT:
				/* Only handle KeyDown events, to avoid duplication */
				if(InRec.Event.KeyEvent.bKeyDown)
				{
					if(InRec.Event.KeyEvent.uChar.AsciiChar)
					{
						Term_keypress(InRec.Event.KeyEvent.uChar.AsciiChar);
					}
					else /* Not an ASCII char - so probably a cursor key or numpad key... */
					{
						switch(InRec.Event.KeyEvent.wVirtualKeyCode)
						{
						case VK_INSERT:
							Term_keypress('0');
							break;
						case VK_END:
							Term_keypress('1');
							break;
						case VK_DOWN:
							Term_keypress('2');
							break;
						case VK_NEXT:
							Term_keypress('3');
							break;
						case VK_LEFT:
							Term_keypress('4');
							break;
						case VK_CLEAR:
							Term_keypress('5');
							break;
						case VK_RIGHT:
							Term_keypress('6');
							break;
						case VK_HOME:
							Term_keypress('7');
							break;
						case VK_UP:
							Term_keypress('8');
							break;
						case VK_PRIOR:
							Term_keypress('9');
							break;
						case VK_DELETE:
							Term_keypress('.');
							break;
						}
					}
				}
			}

			return (0);
		}

		case TERM_XTRA_FLUSH:
		{
			/*
			 * Flush all pending events
			 */
			do
			{
				GetNumberOfConsoleInputEvents(td->hIn,&EventCount);
				if(EventCount)
				{
					/* Throw event away */
					ReadConsoleInput(td->hIn, &InRec, 1, &EventCount);
				}
			} while (EventCount);
			return (0);
		}

		case TERM_XTRA_CLEAR:
		{
			/*
			 * Clear the entire window XXX XXX XXX
			 */
    
			/* Write space characters to buffer as many times as needed, i.e. 80*24 */
			FillConsoleOutputCharacterA(td->hOut, ' ', 2160, TopLeft, &Written);
    
			return (0);
		}

		case TERM_XTRA_SHAPE:
		{
			/*
			 * Set the cursor visibility
			 *
			 * This action changes the visibility of the cursor,
			 * if possible, to the requested value (0=off, 1=on)
			 */

			if(v)
			{
				CursorInfo.bVisible=TRUE;
			}
			else
			{
				CursorInfo.bVisible=FALSE;
			}
			CursorInfo.dwSize=10;
			SetConsoleCursorInfo(td->hOut,&CursorInfo);

			return (0);
		}

		case TERM_XTRA_FROSH:
		{
			/*
			 * Flush a row of output
			 *
			 * This event is ignored, since we handle all
			 * output immediately
			 */

			return (0);
		}

		case TERM_XTRA_FRESH:
		{
			/*
			 * Flush output
			 *
			 * This event is ignored, since we handle all
			 * output immediately
			 */

			return (0);
		}

		case TERM_XTRA_NOISE:
		{
			/*
			 * Make a noise
			 *
			 * a fifth of a second at e-flat. Approximately the standard
			 * pitch and duration of a Taeniopygia guttata call.
			 */
			Beep(311,200);
			return (0);
		}

		case TERM_XTRA_BORED:
		{
			/*
			 * Handle random events when bored XXX XXX XXX
			 *
			 * This action is ignored.
			 */

			return (0);
		}

		case TERM_XTRA_REACT:
		{
			/*
			 * React to global changes
			 *
			 * This action is ignored, since we have nothing
			 * to react to.
			 */

			return (0);
		}

		case TERM_XTRA_ALIVE:
		{
			/*
			 * Change the "hard" level
			 *
			 * This action is ignored, since we have no need
			 * to do anything when our state changes.
			 */

			return (0);
		}

		case TERM_XTRA_LEVEL:
		{
			/*
			 * Change the "soft" level
			 *
			 * This action is ignored, since we have no need
			 * to do anything when our state changes.
			 */

			return (0);
		}

		case TERM_XTRA_DELAY:
		{
			/*
			 * Delay for some milliseconds XXX XXX XXX
			 *
			 * This action is useful for proper "timing" of certain
			 * visual effects, such as breath attacks.
			 */

			Sleep(v);

			return (0);
		}
	}

	/* Unknown or Unhandled action */
	return (1);
}


/*
 * Display the cursor
 */
static errr Term_curs_wcn(int x, int y)
{
	COORD cursorPos = {x,y};

	term_data *td = (term_data*)(Term->data);

	SetConsoleCursorPosition(td->hOut,cursorPos);

	/* Success */
	return (0);
}


/*
 * Erase some characters
 *
 * This function erases "n" characters starting at (x,y).
 */
static errr Term_wipe_wcn(int x, int y, int n)
{
	COORD pos = {x,y};
	DWORD Written;

	term_data *td = (term_data*)(Term->data);

	FillConsoleOutputCharacterA(td->hOut,' ',n,pos,&Written);

	/* Success */
	return (0);
}


/*
 * Draw some text on the screen
 *
 * This function actually displays an array of characters
 * starting at the given location, using the given "attribute",
 * and using the given string of characters, which contains
 * exactly "n" characters and which is NOT null-terminated.
 */
static errr Term_text_wcn(int x, int y, int n, byte a, const char *cp)
{
	DWORD written;
	COORD cursorPos = {x,y};

	term_data *td = (term_data*)(Term->data);

	SetConsoleTextAttribute(td->hOut,color_data[a]);
	SetConsoleCursorPosition(td->hOut,cursorPos);
	WriteConsoleA(td->hOut,cp,n,&written,0);

	/* Success */
	return (0);
}


/*
 * Draw some attr/char pairs on the screen
 *
 * This routine is ignored since we do not have
 * graphical capability
 */
static errr Term_pict_wcn(int x, int y, int n, const byte *ap, const char *cp,
                          const byte *tap, const char *tcp)
{
	term_data *td = (term_data*)(Term->data);

	/* Success */
	return (0);
}



/*** Internal Functions ***/


/*
 * Instantiate a "term_data" structure
 */
static void term_data_link(int i)
{
	term_data *td = &data[i];
	term *t = &td->t;

	/* Initialize the term */
	term_init(t, 80, 24, 256);

	/* Prepare the init/nuke hooks */
	t->init_hook = Term_init_wcn;
	t->nuke_hook = Term_nuke_wcn;

	/* Prepare the template hooks */
	t->user_hook = Term_user_wcn;
	t->xtra_hook = Term_xtra_wcn;
	t->curs_hook = Term_curs_wcn;
	t->wipe_hook = Term_wipe_wcn;
	t->text_hook = Term_text_wcn;
	t->pict_hook = Term_pict_wcn;

	/* Remember where we came from */
	t->data = td;

	/* Activate it */
	Term_activate(t);

	/* Global pointer */
	angband_term[i] = t;
}


/*
 * Help message.
 *   1st line = max 68 chars.
 *   Start next lines with 11 spaces
 */
const char help_wcn[] = "Windows (Win32) Console Version";


/*
 * Initialization function
 */
errr init_wcn(int argc, char **argv)
{
	int i;

	/* Initialize globals */

	SetConsoleTitleA(VERSION_NAME);


	/*
	 * Initialize the "color_data" array.
	 *
	 * We have a very limited colour palette of sixteen colours, so we have
	 * to match these as best we can to Angband's colours.
	 *
	 * This results in a couple of less than perfect matches:
	 *
	 * TERM_SLATE is actually a dark cyan rather than a grey.
	 * TERM_ORANGE is actually a bright red rather than an orange.
	 * TERM_UMBER and TERM_L_UMBER are the same shade.
	 * TERM_L_RED is actually a bright magenta.
	 *
	 * These are the same compromises that 'main_win.c' makes, so the windows
	 * version and the windows console version should look similar.
	 */

	color_data[TERM_DARK]		=	0;						/* 0,0,0 -> 0,0,0 */
	color_data[TERM_WHITE]		=	FOREGROUND_RED |
									FOREGROUND_GREEN |
									FOREGROUND_BLUE |
									FOREGROUND_INTENSITY;	/* 4,4,4 -> 4,4,4 */
	color_data[TERM_SLATE]		=	FOREGROUND_GREEN |
									FOREGROUND_BLUE;		/* 2,2,2 -> 0,2,2 */
	color_data[TERM_ORANGE]		=	FOREGROUND_RED |
									FOREGROUND_INTENSITY;	/* 4,2,0 -> 4,0,0 */
	color_data[TERM_RED]		=	FOREGROUND_RED;			/* 3,0,0 -> 2,0,0 */
	color_data[TERM_GREEN]		=	FOREGROUND_GREEN;		/* 0,2,1 -> 0,2,0 */
	color_data[TERM_BLUE]		=	FOREGROUND_BLUE |
									FOREGROUND_INTENSITY;	/* 0,0,4 -> 0,0,4 */
	color_data[TERM_UMBER]		=	FOREGROUND_RED |
									FOREGROUND_GREEN;		/* 2,1,0 -> 2,2,0 */
	color_data[TERM_L_DARK]		=	FOREGROUND_INTENSITY;	/* 1,1,1 -> 1,1,1 */
	color_data[TERM_L_WHITE]	=	FOREGROUND_RED |
									FOREGROUND_GREEN |
									FOREGROUND_BLUE;		/* 3,3,3 -> 2,2,2 */
	color_data[TERM_VIOLET]		=	FOREGROUND_RED |
									FOREGROUND_BLUE;		/* 4,0,4 -> 2,0,2 */
	color_data[TERM_YELLOW]		=	FOREGROUND_RED |
									FOREGROUND_GREEN |
									FOREGROUND_INTENSITY;	/* 4,4,0 -> 4,4,0 */
	color_data[TERM_L_RED]		=	FOREGROUND_RED | 
									FOREGROUND_BLUE | 
									FOREGROUND_INTENSITY;	/* 4,0,0 -> 4,0,4 */
	color_data[TERM_L_GREEN]	=	FOREGROUND_GREEN | 
									FOREGROUND_INTENSITY;	/* 0,4,0 -> 0,4,0 */
	color_data[TERM_L_BLUE]		=	FOREGROUND_BLUE | 
									FOREGROUND_GREEN | 
									FOREGROUND_INTENSITY;	/* 0,4,4 -> 0,4,4 */
	color_data[TERM_L_UMBER]	=	FOREGROUND_RED | 
									FOREGROUND_GREEN;		/* 3,2,1 -> 2,2,0 */

	/* Initialize "term_data" structures */
	
	/* Create windows (backwards!) */
	for (i = MAX_WCN_TERM; i-- > 0; )
	{
		/* Link */
		term_data_link(i);
	}

	/* Trap console events */
	SetConsoleCtrlHandler(TrapCtrlHandler, TRUE);


	/* Success */
	return (0);
}

#endif /* USE_WCN */